﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.UI;

//PageDefinition is a script used to define a game object as a page
public class PageDefinition : MonoBehaviour
{
    public string Title;    //Title of the page
    public GameObject PageGameObject;   //The root parent game object of the page
    public Button BackButton;   //The page's back navigation button
    public UnityEngine.UI.Selectable DefaultFocusedControl; //The control that should be focused by default

    //Page animations
    public Animator Animator;
    public AnimationClip TransToAnimation;
    public AnimationClip TransFromAnimation;
    public AnimationClip TransBackToAnimation;
    public AnimationClip TransBackFromAnimation;

    private TaskCompletionSource<bool> AnimCompleteTask;  //Task to allow waiting of page to close

    [HideInInspector]
    public GameObject PreviousFocusedGameObject;

    public void Awake()
    {
        AnimCompleteTask = new TaskCompletionSource<bool>();

        Animator animator = gameObject.GetComponent<Animator>();
        AnimatorOverrideController controller = new AnimatorOverrideController(animator.runtimeAnimatorController);
        animator.runtimeAnimatorController = controller;
        controller["TransTo"] = TransToAnimation;
        controller["TransFrom"] = TransFromAnimation;
        controller["TransBackTo"] = TransBackToAnimation;
        controller["TransBackFrom"] = TransBackFromAnimation;
    }

    /// <summary>
    /// Refocuses to the appropriate page control, defaulting to the previous one first and then falling back to the default
    /// </summary>
    public void ReFocus()
    {
        if (PreviousFocusedGameObject != null)
        {
            EventSystem.current.SetSelectedGameObject(PreviousFocusedGameObject);
        }

        else if (DefaultFocusedControl != null)
        {
            EventSystem.current.SetSelectedGameObject(DefaultFocusedControl.gameObject);
        }

        else
        {
            Debug.Log("Couldn't refocus!");
        }
    }

    /// <summary>
    /// Play an animation of animName, returning when the animation completes
    /// </summary>
    /// <param name="animName">The name of the animation to play</param>
    /// <returns></returns>
    private async Task PlayAnimation(string animName)
    {
        try
        {
            Animator.enabled = true;
            Animator.Play(animName, 0, 0.0f);
            await AnimCompleteTask.Task;
            AnimCompleteTask = new TaskCompletionSource<bool>();
            Animator.enabled = false;
        }

        catch (Exception ex)
        {
            Debug.Log("Exception: " + ex);
        }
    }

    /// <summary>
    /// Performs the TransTo animation and plays the TransTo sound, returning when the animation completes
    /// </summary>
    /// <returns></returns>
    public async Task TransTo()
    {
        if (AudioManager.Instance != null && AudioManager.Instance.LoadedFiles[Constants.UIPageTransToAudioID].Clip != null)
        {
            //We have a manager and audio, play the trans to audio
            AudioManager.Instance.PlaySFX(AudioManager.Instance.LoadedFiles[Constants.UIPageTransToAudioID].Clip);
        }

        await PlayAnimation("TransTo");
    }

    /// <summary>
    /// Performs the TransFrom animation and plays the TransFrom sound, returning when the animation completes
    /// </summary>
    /// <returns></returns>
    public async Task TransFrom()
    {
        if (AudioManager.Instance != null && AudioManager.Instance.LoadedFiles[Constants.UIPageTransFromAudioID].Clip != null)
        {
            //We have a manager and audio, play the trans from audio
            AudioManager.Instance.PlaySFX(AudioManager.Instance.LoadedFiles[Constants.UIPageTransFromAudioID].Clip);
        }

        await PlayAnimation("TransFrom");
    }

    /// <summary>
    /// Performs the TransBackTo animation and plays the TransBackTo sound, returning when the animation completes
    /// </summary>
    /// <returns></returns>
    public async Task TransBackTo()
    {
        if (AudioManager.Instance != null && AudioManager.Instance.LoadedFiles[Constants.UIPageTransBackToAudioID].Clip != null)
        {
            //We have a manager and audio, play the trans back to audio
            AudioManager.Instance.PlaySFX(AudioManager.Instance.LoadedFiles[Constants.UIPageTransBackToAudioID].Clip);
        }

        await PlayAnimation("TransBackTo");
    }

    /// <summary>
    /// Performs the TransBackFrom animation and plays the TransBackFrom sound, returning when the animation completes
    /// </summary>
    /// <returns></returns>
    public async Task TransBackFrom()
    {
        if (AudioManager.Instance != null && AudioManager.Instance.LoadedFiles[Constants.UIPageTransBackFromAudioID].Clip != null)
        {
            //We have a manager and audio, play the trans back from audio
            AudioManager.Instance.PlaySFX(AudioManager.Instance.LoadedFiles[Constants.UIPageTransBackFromAudioID].Clip);
        }

        await PlayAnimation("TransBackFrom");
    }

    /// <summary>
    /// Handler for when the transition animation completes, setting the task to allow execution to continue after await
    /// </summary>
    public void OnTransitionAnimationCompleted()
    {
        AnimCompleteTask.SetResult(true);
    }
}
